using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using EzeScan.Interfaces;

namespace EDRMS_Output
{
    public class Output_Class : IOutput
    {
        private UserConfigurableParameter[] _outputParameters;

        public Output_Class()
        {
            // create an array of values to be used by the lookup parameter
            String[] no_yes = new String[] {"No", "Yes"};
        
            // Create the output parameters            
            UserConfigurableParameter new_parameter = Connection_Class.PluginHelper.CreateLookupParameter("inc", "Increment Doc ID", ParameterType.Lookup, no_yes[1], "Increment the Document ID", ref no_yes, false);
            Connection_Class.PluginHelper.AddParameter(ref _outputParameters, ref new_parameter);
        }


        #region IOutput Members

        int IOutput.EditCustomParameter(string parameter_id, ref UserConfigurableParameter[] output_parameters, ref string value_out)
        {
            // See Connection_Class.IConnection.EditCustomParameter() for a description of this function
            // This plugin output object does not contain any custom parameters - so the return code is always cancel just in case this function is ever called in error
            return 0;
        }

        int IOutput.OutputDocument(string document_path, ref UserConfigurableParameter[] output_parameters, string previous_document_reference)
        {
            try
            {
                Boolean increment_id = false;
                String document_id = (previous_document_reference == null ? "" : previous_document_reference);
    
                // Process the parameters
                
                for (Int32 param_index = 0; param_index < Connection_Class.PluginHelper.ParameterCount(ref output_parameters); param_index++)
                {
    
                    switch (output_parameters[param_index].ID)
                    {
                        case "inc":
                            increment_id = (output_parameters[param_index].Value == "Yes");
                            Connection_Class.DebugLog.LogPluginMessage( "Increment ID: " + (increment_id ? "Yes" : "No"), PluginDebugLevel.InfoNormal);
                            break;
                    }
                    
                }
    

                if (document_id != "")
                {
                    DialogResult user_response = MessageBox.Show("Replace existing document with new version?", "Replace Document", MessageBoxButtons.YesNoCancel);

                    switch (user_response)
                    {
                        case DialogResult.Yes:
                            Connection_Class.DebugLog.LogPluginMessage( "Add new version of: " + document_id, PluginDebugLevel.InfoNormal);
                            break;
                        case DialogResult.No:
                            Connection_Class.DebugLog.LogPluginMessage( "Ignoring previous document reference: " + document_id, PluginDebugLevel.InfoNormal);
                            document_id = "";
                            break;
                        case DialogResult.Cancel:
                            Connection_Class.DebugLog.LogPluginMessage( "Cancelling Output", PluginDebugLevel.InfoNormal);
                            return 0;
                    }                
                }
    
                String base_folder = System.IO.Path.Combine(Connection_Class.StorageFolder, Connection_Class.UserName);
        
                if (document_id == "")
                {
                    document_id = "1";
                    Int32 next_id = 1;

                    while (increment_id && System.IO.File.Exists(System.IO.Path.Combine(base_folder, document_id + System.IO.Path.GetExtension(document_path))))
                    {
                        next_id++; 
                        document_id = next_id.ToString();
                    }
                    
                    Connection_Class.DebugLog.LogPluginMessage( "Using new document reference: " + document_id, PluginDebugLevel.InfoNormal);
                }
    
                if (!System.IO.Directory.Exists(base_folder))
                {
                    Connection_Class.DebugLog.LogPluginMessage( "Creating Folder: " + base_folder, PluginDebugLevel.InfoNormal);
                    System.IO.Directory.CreateDirectory( base_folder);
                }
    
                String target_file_path = System.IO.Path.Combine(base_folder, document_id + System.IO.Path.GetExtension(document_path));
    
                if (System.IO.File.Exists(target_file_path))
                {
                    Connection_Class.DebugLog.LogPluginMessage( "Removing existing file: " + target_file_path, PluginDebugLevel.InfoNormal);
                    System.IO.File.Delete(target_file_path);
                }
    
                Connection_Class.DebugLog.LogPluginMessage( "Saving file: " + target_file_path, PluginDebugLevel.InfoNormal);
                System.IO.File.Copy(document_path, target_file_path, false);
        
                Connection_Class.PreviousDocumentReference = document_id;
                return 1; // success
    
            }
            catch(Exception ex)
            {
                Connection_Class.DebugError(ex);
                Connection_Class.DebugLog.LogUserErrorMessage("Output error: " + ex.Message);
                return -99;  // error
            }
        }

        UserConfigurableParameter[] IOutput.OutputParameters
        {
            get 
            { 
                // EzeScan calls this property to retrieve an array of parameters that are required to output a document
                return _outputParameters; 
            }
        }

        #endregion
    }
}
